<?php

namespace Welford\HttpRateLimiter\Adapters;

use Exception;
use PDO as PhpPDO;

class PDO extends BaseAdapter
{

    /**
     * @throws Exception
     */
    public static function getCount(int $expire = 60, array $options = []): int
    {
        if (!extension_loaded('pdo')) {
            throw new Exception('PDO extension not loaded');
        }

        if (empty($options['type'])) {
            throw new Exception('PDO type not set');
        }

        switch ($options['type']) {
            case 'mysql':
                if (empty($options['host']) || empty($options['port']) || empty($options['database']) || empty($options['username']) || empty($options['password']) || empty($options['table'])) {
                    throw new Exception('PDO MySQL options are not all set. Required options are: host, port, database, username, password and table.');
                }
                $dsn = "mysql:host={$options['host']};port={$options['port']};dbname={$options['database']}";
                break;
            case 'pgsql':
                if (empty($options['host']) || empty($options['port']) || empty($options['database']) || empty($options['username']) || empty($options['password']) || empty($options['table'])) {
                    throw new Exception('PDO PostgreSQL options are not all set. Required options are: host, port, database, username, password and table.');
                }
                $dsn = "pgsql:host={$options['host']};port={$options['port']};dbname={$options['database']}";
                break;
            case 'sqlite':
                if (empty($options['database'])) {
                    throw new Exception('PDO SQLite database not set');
                } elseif (!file_exists($options['database'])) {
                    throw new Exception('PDO SQLite database does not exist');
                } elseif (!is_writable($options['database'])) {
                    throw new Exception('PDO SQLite database is not writable');
                } elseif (empty($options['table'])) {
                    throw new Exception('PDO SQLite table not set');
                }
                $dsn = "sqlite:{$options['database']}";
                break;
            default:
                throw new Exception('PDO type not supported. Supported types are: mysql, pgsql and sqlite.');
        }

        $pdo = new PhpPDO($dsn, $options['username'] ?? null, $options['password'] ?? null);
        $pdo->setAttribute(PhpPDO::ATTR_ERRMODE, PhpPDO::ERRMODE_EXCEPTION);
        $pdo->exec("CREATE TABLE IF NOT EXISTS {$options['table']} (cache_key TEXT NOT NULL, cache_timestamp BIGINT NOT NULL);");

        $key = self::getCacheKey();
        $now = time();

        // Insert this visit into table
        $query = "INSERT INTO {$options['table']} (`cache_key`, `cache_timestamp`) VALUES (:key, :timestamp)";
        $stmt = $pdo->prepare($query);
        $stmt->execute(['key' => $key, 'timestamp' => $now]);

        // Cleanup rows that are older than $expire
        $query = "DELETE FROM {$options['table']} WHERE `cache_key` = :key AND `cache_timestamp` < :timestamp";
        $stmt = $pdo->prepare($query);
        $stmt->execute(['key' => $key, 'timestamp' => $now - $expire]);

        // Fetch count of rows for key which are within expire time
        $query = "SELECT COUNT(*) as total FROM {$options['table']} WHERE `cache_key` = :key AND `cache_timestamp` >= :timestamp";
        $stmt = $pdo->prepare($query);
        $stmt->execute(['key' => $key, 'timestamp' => $now - $expire]);

        return $stmt->fetchColumn() ?: 0;
    }
}
