<?php

namespace Welford\HttpRateLimiter;

use Exception;
use Welford\HttpRateLimiter\Adapters\File;
use Welford\HttpRateLimiter\Adapters\Memcached;
use Welford\HttpRateLimiter\Adapters\PDO;
use Welford\HttpRateLimiter\Adapters\Redis;

class HttpRateLimiter {
    const REDIS = 'redis';
    const MEMCACHED = 'memcached';
    const PDO = 'pdo';
    const FILE = 'file';

    /**
     * Check if the request IP address has exceeded the rate limit.
     *
     * @param string $connectionType The type of connection to use. Can be 'redis', 'memcached', 'pdo' or 'file'.
     * @param array $connectionOptions An array of options to pass to the connection adapter.
     * @param int $limit The number of requests allowed in the time period.
     * @param int $expire The time in seconds that the rate limit will be applied to.
     * @throws Exception
     */
    public static function check(string $connectionType, array $connectionOptions, int $limit = 30, int $expire = 60): void
    {
        $count = 0;
        switch ($connectionType) {
            case self::REDIS:
                $count = Redis::getCount($expire, $connectionOptions);
                break;
            case self::MEMCACHED:
                $count = Memcached::getCount($expire, $connectionOptions);
                break;
            case self::PDO:
                $count = PDO::getCount($expire, $connectionOptions);
                break;
            case self::FILE:
                $count = File::getCount($expire, $connectionOptions);
                break;
            default:
                throw new Exception('Invalid connection type');
        }
        if ($count > $limit) {
            ob_clean();
            ob_start();
            header('HTTP/1.1 429 Too Many Requests');
            echo 'Too many requests';
            ob_end_flush();
            die();
        }
    }

}
