<?php

namespace Welford\Ipfs;

use Exception;

class Client
{

    protected string $host;
    protected string $port;

    public function __construct(string $host = "127.0.0.1", string $port = "5001")
    {
        $this->host = $host;
        $this->port = $port;
    }

    /**
     * Add a single file to IPFS. Allows for directory wrapping to expose filename.
     * Files are not pinned by default.
     * @param string $absolutePath
     * @param bool $wrapInDirectory
     * @return array
     * @throws Exception
     */
    public function addFile(string $absolutePath, bool $wrapInDirectory = false): array
    {
        $boundary = uniqid();

        if (!is_dir($absolutePath)) {
            // Upload as single file

            $parts = explode(DIRECTORY_SEPARATOR, $absolutePath);
            $file = [[
                'name' => $parts[count($parts) - 1],
                'path' => $absolutePath
            ]];

            $request_body = $this->constructMultipart(null, $file, $boundary);
            $output = $this->upload($request_body, $boundary, $wrapInDirectory);
            if (!is_string($output)) {
                return $output;
            }

            $lines = array_filter(explode("\n", $output));
            $hashes = [];
            foreach ($lines as $line) {
                $hashes[] = json_decode($line, true);
            }

            return ($wrapInDirectory) ? ['hash' => $hashes[0]['Hash'] . '/' . $file[0]['name']] : ['hash' => $hashes[0]['Hash']];
        }

        throw new Exception('Path is not a file');
    }

    /**
     * Adds a folder to IPFS (non-recursive)
     * As per IPFS docs, files stored inside folders are pinned by default
     * through recursion.
     * @param string $absolutePath
     * @return array|bool|string
     * @throws Exception
     */
    public function addFolder(string $absolutePath)
    {
        $boundary = uniqid();

        if (is_dir($absolutePath)) {
            // Upload as directory (non-recursive)

            $files = [];
            foreach ($this->getDirectoryFiles($absolutePath) as $filePath) {
                $parts = explode(DIRECTORY_SEPARATOR, $filePath);
                $files[] = [
                    'name' => $parts[count($parts) - 1],
                    'path' => $filePath
                ];
            }
            $request_body = $this->constructMultipart($boundary, $files, $boundary);
            unset($files);

            $output = $this->upload($request_body, $boundary);
            if (!is_string($output)) {
                return $output;
            }

            $lines = array_filter(explode("\n", $output));
            $hashes = [];
            foreach ($lines as $line) {
                $hashes[] = json_decode($line, true);
            }
            $key = array_search($boundary, array_column($hashes, 'Name'));

            return ['hash' => $hashes[$key]['Hash']];
        }

        throw new Exception('Path is not a folder');
    }

    /**
     * Pins an IPFS hash to the servers local pinning service
     * @param string $hash
     * @return array
     */
    public function pin(string $hash): array
    {
        $params = [
            'arg' => $hash
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "http://" . $this->host . ':' . $this->port . '/api/v0/pin/add?' . http_build_query($params),
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 7200,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POST => 1,
            CURLOPT_HTTPHEADER => ["Accept: application/json"]
        ]);

        curl_exec($curl);

        $error = curl_error($curl);
        if (!empty($error)) {
            return ['curl_error' => $error];
        }

        curl_close($curl);
        return ['hash' => $hash];
    }

    private function upload(string $request_body, string $boundary, bool $wrap_in_directory = false)
    {
        $headers = [
            "Content-Type: multipart/form-data; boundary=" . '-------------' . $boundary,
            "Content-Length: " . strlen($request_body),
            "Accept: application/json"
        ];

        $params = [
            'quiet' => true,
            'quieter' => true,
            'wrap-with-directory' => $wrap_in_directory
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "http://" . $this->host . ':' . $this->port . '/api/v0/add?' . http_build_query($params),
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 7200,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POST => 1,
            CURLOPT_POSTFIELDS => $request_body,
            CURLOPT_HTTPHEADER => $headers
        ]);

        $output = curl_exec($curl);

        $error = curl_error($curl);
        if (!empty($error)) {
            return ['curl_error' => $error];
        }

        curl_close($curl);

        return $output;
    }

    private function constructMultipart($directory, array $files, string $boundary): string
    {
        $delimiter = '-------------' . $boundary;
        $data = '';
        $eol = "\r\n";

        if ($directory !== null) {
            $data .= "--" . $delimiter . $eol
                . 'Content-Disposition: form-data; name="file"; filename="' . $directory . '"' . $eol
                . 'Content-Type: application/x-directory' . $eol . $eol;
        }

        foreach ($files as $file) {
            $filename = ($directory !== null) ? $directory . '/' . $file['name'] : $file['name'];
            $data .= "--" . $delimiter . $eol
                . 'AbsPath:' . $file['path'] . $eol
                . 'Content-Disposition: form-data; name="file"; filename="' . $filename . '"' . $eol
                . 'Content-Type: application/octet-stream' . $eol
            ;
            $data .= $eol;
            $data .= file_get_contents($file['path']) . $eol;
        }
        $data .= "--" . $delimiter . "--".$eol;

        return $data;
    }

    private function getDirectoryFiles($dir): array
    {
        $results = array();
        $files = scandir($dir);
        foreach ($files as $key => $value) {
            $path = realpath($dir . DIRECTORY_SEPARATOR . $value);
            if (!is_dir($path)) {
                $results[] = $path;
            }
        }
        return $results;
    }

}
