<?php

namespace Welford\NhsMeshLaravel;

use Exception;
use Illuminate\Mail\Transport\Transport;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Welford\NhsMesh\Client;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Swift_Mime_SimpleMessage;
use ZBateson\MailMimeParser\Message;
use ZipArchive;

class NhsMeshTransport extends Transport
{

    private Client $meshClient;

    private string $mailbox_id = "";
    private string $mailbox_password = "";
    private int $environment = Client::DEV;
    private string $hmac_secret = "";

    public function __construct(
        string $ca_cert,
        string $mailbox_cert,
        string $mailbox_key,
        string $mailbox_id,
        string $mailbox_password,
        int $environment,
        string $hmac_secret)
    {
        $this->mailbox_id = $mailbox_id;
        $this->mailbox_password = $mailbox_password;
        $this->environment = $environment;
        $this->hmac_secret = $hmac_secret;

        file_put_contents('/tmp/nhs-mesh-laravel-ca.pem', $ca_cert);
        file_put_contents('/tmp/nhs-mesh-laravel-cert.pem', $mailbox_cert);
        file_put_contents('/tmp/nhs-mesh-laravel-key.pem', $mailbox_key);

        $this->meshClient = new Client(
            $this->mailbox_id,
            $this->mailbox_password,
            $this->environment,
            $this->hmac_secret,
            '/tmp/nhs-mesh-laravel-ca.pem',
            '/tmp/nhs-mesh-laravel-key.pem',
            '/tmp/nhs-mesh-laravel-cert.pem');
    }

    /**
     * {@inheritdoc}
     */
    public function send(Swift_Mime_SimpleMessage $message, &$failedRecipients = null)
    {
        $this->beforeSendPerformed($message);

        $id = uniqid();
        Storage::makeDirectory($id);

        $rawMessage = $message->toString();

        /* 
            To addresses should be formatted as {mailbox_id}@{workflow_id}.com
            The addresses will the be parsed into the required headers for transmission
        */

        $parsedMessage = Message::from($rawMessage, false);

        $to = $parsedMessage->getHeader('To');
        $addresses = [];
        foreach ($to->getAddresses() as $address) {
            $to_parts = explode("@", $address->getEmail());
            if (count($to_parts) == 2) {
                $addresses[] = [
                    'mailbox_id' => $to_parts[0],
                    'workflow_id' => str_replace(".com", "", $to_parts[1])
                ];
            }
        }

        /*
            The plain text body of the email is converted to a TXT file,
            The html body of the email is converted into a .html document and
            attachments from the email will be extracted and compressed together 
            into a single Zip file
        */

        $plainBody = $parsedMessage->getTextContent();
        $htmlBody = $parsedMessage->getHtmlContent();

        if (!empty($plainBody)) {
            Storage::put($id . '/plain_message.txt', $plainBody);
        }

        if (!empty($htmlBody)) {
            Storage::put($id . '/html_message.html', $htmlBody);
        }

        $atts = $parsedMessage->getAllAttachmentParts();
        foreach ($atts as $ind => $part) {
            try {
                $filename = $part->getFilename();
                $out = fopen(storage_path($id . '/' . $filename), 'w');
                $str = $part->getBinaryContentResourceHandle();
                stream_copy_to_stream($str, $out);
                fclose($str);
                fclose($out);
            } catch (Exception $e) {
                Log::error('Could not save attachment for email : ' . $e->getMessage());
            }
        }

        $zip = new ZipArchive();
        $zip->open(storage_path('app/' . $id . '.zip'), ZipArchive::CREATE | ZipArchive::OVERWRITE);
        
        $path = storage_path('app/' . $id);
        $files = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path));
        foreach ($files as $name => $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath =  '/' . substr($filePath, strlen($path) + 1);
                $zip->addFile($filePath, $relativePath);
            }
        }
        $zip->close();
        
        foreach($addresses as $address) {
            $this->meshClient->sendMessage(
                $address['mailbox_id'],
                $address['workflow_id'],
                $id . '.zip',
                file_get_contents(storage_path('app/' . $id . '.zip')),
                $message->getSubject());
        }

        Storage::delete($id . '.zip');
        Storage::deleteDirectory($id);

        $this->sendPerformed($message);

        return $this->numberOfRecipients($message);
    }

}