<?php

namespace Welford\NhsMesh;

use DateTime;
use GuzzleHttp\Client as GuzzleHttpClient;
use Welford\NhsMesh\Exceptions\InvalidEnvironmentException;
use Welford\NhsMesh\Requests\LookupMailboxAddressRequest;
use Welford\NhsMesh\Requests\SendMessageRequest;
use Welford\NhsMesh\Requests\ValidateMailboxRequest;
use Welford\NhsMesh\Responses\LookupMailboxAddressResponse;
use Welford\NhsMesh\Responses\SendMessageResponse;
use Welford\NhsMesh\Responses\ValidateMailboxResponse;

class Client {

    public const PROD = 1;
    public const DEV = 2;
    private const PROD_URL = "https://mesh-sync.spineservices.nhs.uk";
    private const DEV_URL = "https://msg.intspineservices.nhs.uk";
    private const AUTH_SCHEMA = "NHSMESH ";

    private GuzzleHttpClient $guzzleClient;
    private string $mailboxId;
    private string $password;
    private string $hmac_secret;

    public function __construct(string $mailboxId, string $password, int $environment, string $hmac_secret, string $ca_cert, string $client_key, string $client_cert)
    {
        $this->mailboxId = $mailboxId;
        $this->password = $password;
        $this->hmac_secret = $hmac_secret;

        $options = [
            'timeout'  => 4.0,
            'headers'  => [
                'Accept'     => 'application/json',
                'Authorization' => $this->generateAuthHeader(),
                'Mex-From' => $mailboxId,
                'Mex-ClientVersion' => 'ApiDocs==0.0.1',
                'Mex-OSName' => php_uname("s"),
                'Mex-OSVersion' => php_uname("v")

            ],
            'curl' => [
                CURLOPT_CAINFO => $ca_cert
            ],
            'cert' => $client_cert,
            'ssl_key' => $client_key
        ];

        switch ($environment) {
            case 1:
                $this->environment = $environment;
                $options['base_uri'] = self::PROD_URL;
                break;
            case 2:
                $this->environment = $environment;
                $options['base_uri'] = self::DEV_URL;
                break;
            default:
                throw new InvalidEnvironmentException("The provided environment was invalid.");
        }

        $this->guzzleClient = new GuzzleHttpClient($options);
    }

    public function validateMailbox(string $mailboxId) {
        $response = new ValidateMailboxResponse(
            new ValidateMailboxRequest(['mailboxId' => $mailboxId]),
            $this->guzzleClient
        );
        return $response;
    }

    public function lookupMailboxAddress(string $ods_code, string $workflow_id) {
        $response = new LookupMailboxAddressResponse(
            new LookupMailboxAddressRequest(['odsCode' => $ods_code, 'workflowId' => $workflow_id]),
            $this->guzzleClient
        );
        return $response;
    }

    public function sendMessage(string $to_mailboxId, string $to_workflowId, string $filename, string $data, string $subject = null) {
        $response = new SendMessageResponse(
            new SendMessageRequest([
                'to_mailboxId' => $to_mailboxId,
                'to_workflowId' => $to_workflowId,
                'from_mailboxId' => $this->mailboxId,
                'filename' => $filename,
                'subject' => $subject,
                'data' => $data
            ]),
            $this->guzzleClient
        );
        return $response;
    }

    private function generateAuthHeader() {
        $uid = uniqid();
        $dateTime = new DateTime();
        $timestamp = $dateTime->format('YmdHi');
        $hmac_msg = $this->mailboxId . ":" . $uid . ":" . "0" . ":" . $this->password . ":" . $timestamp;
        $hmac_sig = hash_hmac('sha256', $hmac_msg, $this->hmac_secret);
        return self::AUTH_SCHEMA . $this->mailboxId . ":" . $uid . ":" . "0" . ":" . $timestamp . ':' . $hmac_sig;
    }

}