<?php

namespace Welford\SimpleCsv;

use Exception;
use PDO;
use stdClass;

class CsvFile
{
    private array $headers = [];
    private array $rows = [];

    /**
     * @return bool
     */
    public function hasHeaders(): bool
    {
        return !empty($this->headers);
    }

    /**
     * @param string $header
     * @return bool
     */
    public function hasHeader(string $header): bool
    {
        return in_array($header, $this->headers);
    }

    /**
     * @return bool
     */
    public function hasRows(): bool
    {
        return !empty($this->rows);
    }

    /**
     * @param int $index
     * @return bool
     */
    public function hasRow(int $index): bool
    {
        return isset($this->rows[$index]);
    }

    /**
     * @return array
     */
    public function getHeaders(): array
    {
        return $this->headers;
    }

    /**
     * @param array $headers
     * @return CsvFile
     */
    public function setHeaders(array $headers): CsvFile
    {
        $this->headers = $headers;
        return $this;
    }

    /**
     * @param bool $withHeaders
     * @return array
     */
    public function getRows(bool $withHeaders = false): array
    {
        if ($withHeaders && $this->hasHeaders()) {
            $rows = [];
            foreach ($this->rows as $row) {
                $new_row = [];
                foreach ($row as $key => $value) {
                    $new_row[$this->headers[$key] ?? $key] = $value;
                }
                $rows[] = $new_row;
            }
            return $rows;
        }
        return $this->rows;
    }

    /**
     * @param array $rows
     * @return CsvFile
     * @throws Exception
     */
    public function setRows(array $rows): CsvFile
    {
        foreach ($rows as $key => $row) {
            if ($this->rowHasHeaders($row))
                $rows[$key] = $this->covertHeadersToColumnIndexes($row);
        }
        $this->rows = $rows;
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * @param int $index
     * @param bool $withHeaders
     * @return array
     */
    public function getRow(int $index, bool $withHeaders = false): array
    {
        if ($withHeaders && $this->hasHeaders()) {
            $new_row = [];
            foreach ($this->rows[$index] as $key => $value) {
                $new_row[$this->headers[$key] ?? $key] = $value;
            }
            return $new_row;
        }
        return $this->rows[$index];
    }

    /**
     * @param int $index
     * @param array $row
     * @return CsvFile
     * @throws Exception
     */
    public function setRow(int $index, array $row): CsvFile
    {
        if ($this->rowHasHeaders($row))
            $row = $this->covertHeadersToColumnIndexes($row);
        $this->rows[$index] = $row;
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * @param array $row
     * @return CsvFile
     * @throws Exception
     */
    public function addRow(array $row): CsvFile
    {
        if ($this->rowHasHeaders($row))
            $row = $this->covertHeadersToColumnIndexes($row);
        $this->rows[] = $row;
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * @param int $index
     * @param array $row
     * @return CsvFile
     * @throws Exception
     */
    public function addRowAtIndex(int $index, array $row): CsvFile
    {
        if ($this->rowHasHeaders($row))
            $row = $this->covertHeadersToColumnIndexes($row);
        array_splice($this->rows, $index, 0, [$row]);
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * @param int $index
     * @return CsvFile
     */
    public function deleteRow(int $index): CsvFile
    {
        array_splice($this->rows, $index, 1);
        return $this;
    }

    /**
     * @param int $columnIndex
     * @param int $rowIndex
     * @return mixed
     * @throws Exception
     */
    public function getCell(int $columnIndex, int $rowIndex)
    {
        if (!isset($this->rows[$rowIndex][$columnIndex])) {
            throw new Exception("The specified cell does not exist");
        }
        return $this->rows[$rowIndex][$columnIndex];
    }

    /**
     * @param string $header
     * @param int $rowIndex
     * @return mixed
     * @throws Exception
     */
    public function getCellByHeader(string $header, int $rowIndex)
    {
        if (!$this->hasHeaders()) {
            throw new Exception("The CSV does not contain any headers");
        }

        if (!in_array($header, $this->headers)) {
            throw new Exception("The specified header does not exist");
        }

        $key = array_keys($this->headers, $header)[0];
        return $this->rows[$rowIndex][$key];
    }

    /**
     * @param int $columnIndex
     * @param int $rowIndex
     * @param string $value
     * @return CsvFile
     */
    public function setCell(int $columnIndex, int $rowIndex, string $value): CsvFile
    {
        if (!isset($this->rows[$rowIndex])) {
            $this->rows[$rowIndex] = [];
        }
        $this->rows[$rowIndex][$columnIndex] = $value;
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * @param string $header
     * @param int $rowIndex
     * @param string $value
     * @return CsvFile
     * @throws Exception
     */
    public function setCellByHeader(string $header, int $rowIndex, string $value): CsvFile
    {
        if (!$this->hasHeaders()) {
            throw new Exception("The CSV does not contain any headers");
        }

        if (!in_array($header, $this->headers)) {
            throw new Exception("The specified header does not exist");
        }
        if (!isset($this->rows[$rowIndex])) {
            $this->rows[$rowIndex] = [];
        }
        $key = array_keys($this->headers, $header)[0];
        $this->rows[$rowIndex][$key] = $value;
        $this->ensureRowLengths();
        return $this;
    }

    /**
     * Searches all columns in the CSV for the specified value
     * @param string $value
     * @param bool $caseSensitive
     * @param bool $exactMatch
     * @param bool $returnHeaders
     * @return array
     */
    public function search(string $value, bool $caseSensitive = false, bool $exactMatch = false, bool $returnHeaders = false): array
    {
        $results = [];
        foreach ($this->rows as $key => $row) {
            foreach ($row as $row_value) {
                if ($caseSensitive) {
                    if ($exactMatch) {
                        if ($row_value == $value) {
                            $results[] = $this->rows[$key];
                        }
                    } else {
                        if (strpos($row_value, $value) !== false) {
                            $results[] = $this->rows[$key];
                        }
                    }
                } else {
                    if ($exactMatch) {
                        if (strtoupper($row_value) == strtoupper($value)) {
                            $results[] = $this->rows[$key];
                        }
                    } else {
                        if (strpos(strtoupper($row_value), strtoupper($value)) !== false) {
                            $results[] = $this->rows[$key];
                        }
                    }
                }
            }
        }

        if ($returnHeaders)
            array_unshift($results, $this->headers);
        return $results;
    }


    /**
     * Searches the specified column name or column index for the specified value
     * @param mixed $headerOrColumnIndex
     * @param string $value
     * @param bool $caseSensitive
     * @param bool $exactMatch
     * @param bool $returnHeaders
     * @return array
     * @throws Exception
     */
    public function searchByColumn($headerOrColumnIndex, string $value, bool $caseSensitive = false, bool $exactMatch = false, bool $returnHeaders = false): array
    {
        if (!is_int($headerOrColumnIndex)) {
            if (!$this->hasHeaders()) {
                throw new Exception("A header was passed but the CSV does not contain any headers");
            }
            if (!in_array($headerOrColumnIndex, $this->getHeaders())) {
                throw new Exception("The CSV does not contain the specified header");
            }
            $headerOrColumnIndex = array_keys($this->headers, $headerOrColumnIndex)[0];
        }

        $results = [];
        foreach ($this->rows as $key => $row) {
            if ($caseSensitive) {
                if ($exactMatch) {
                    if ($row[$headerOrColumnIndex] == $value) {
                        $results[] = $this->rows[$key];
                    }
                } else {
                    if (strpos($row[$headerOrColumnIndex], $value) !== false) {
                        $results[] = $this->rows[$key];
                    }
                }
            } else {
                if ($exactMatch) {
                    if (strtoupper($row[$headerOrColumnIndex]) == strtoupper($value)) {
                        $results[] = $this->rows[$key];
                    }
                } else {
                    if (strpos(strtoupper($row[$headerOrColumnIndex]), strtoupper($value)) !== false) {
                        $results[] = $this->rows[$key];
                    }
                }
            }
        }

        if ($returnHeaders)
            array_unshift($results, $this->headers);
        return $results;
    }

    /**
     * @param array $data
     * @param int $pdoQueryType
     * @return CsvFile
     * @throws Exception
     */
    public function createFromPDOFetchResult(array $data, int $pdoQueryType): CsvFile
    {
        if (!class_exists('PDO')) {
            throw new Exception("The PDO extension is not installed or not enabled");
        }

        if (!in_array($pdoQueryType, [PDO::FETCH_ASSOC, PDO::FETCH_UNIQUE])) {
            throw new Exception("Unsupported query type");
        }

        $this->setHeaders([]);
        $this->setRows([]);

        switch ($pdoQueryType) {
            case PDO::FETCH_ASSOC:
                foreach ($data as $row) {
                    if (!$this->hasHeaders())
                        $this->setHeaders(array_keys($row));
                    $this->addRow($row);
                }
                break;
            case PDO::FETCH_UNIQUE:
                foreach ($data as $key => $row) {
                    if (!$this->hasHeaders()) {
                        $keys = array_keys($row);
                        array_unshift($keys, "id");
                        $this->setHeaders($keys);
                    }
                    array_unshift($row, $key);
                    $this->addRow($row);
                }
                break;
        }

        return $this;
    }

    /**
     * @param stdClass $result
     * @param array|null $headers
     * @return CsvFile
     * @throws Exception
     */
    public function createFromOpenCartQuery(stdClass $result, array $headers = null): CsvFile
    {
        if (!isset($result->rows)) {
            throw new Exception("The results object does not contain the rows property");
        }

        $this->setHeaders([]);
        $this->setRows([]);

        foreach ($result->rows as $row) {
            if (!$this->hasHeaders() && $this->rowHasHeaders($row))
                $this->setHeaders(array_keys($row));
            $this->addRow($row);
        }

        if (!$this->hasHeaders() && !empty($headers)) {
            $this->setHeaders($headers);
        }

        return $this;
    }

    private function ensureRowLengths()
    {
        $max = 0;
        foreach ($this->rows as $key => $row) {
            // Remove blank columns
            $this->rows[$key] = array_filter($row);

            // Calculate the highest key from all rows
            $cols = max(array_keys($row));
            if ($max < $cols)
                $max = $cols;
        }

        // Add missing  keys on all rows up to max cols
        foreach ($this->rows as $key => $row) {
            for ($i = 0; $i <= $max; $i++) {
                if (!isset($this->rows[$key][$i])) {
                    $this->rows[$key][$i] = null;
                }
            }
        }
    }

    /**
     * @throws Exception
     */
    private function covertHeadersToColumnIndexes(array $row): array
    {
        $new_row = [];
        if (!$this->hasHeaders()) {
            throw new Exception("A row was passed with headers but the CSV does not contain any headers");
        }
        foreach ($row as $key => $value) {
            if (!$this->hasHeader($key)) {
                throw new Exception("A row was passed with an invalid header");
            }
            $headerKey = array_keys($this->headers, $key)[0];
            $new_row[$headerKey] = $value;
        }
        ksort($new_row, SORT_NATURAL);
        return array_values($new_row);
    }

    private function rowHasHeaders(array $row): bool
    {
        return count(array_filter(array_keys($row), 'is_string')) > 0;
    }


}
