<?php

use PHPUnit\Framework\TestCase;
use Welford\SimpleCsv\CsvFile;
use Welford\SimpleCsv\SimpleCsv;

class CsvFileTest extends TestCase
{
    private string $filename = __DIR__ . '/test.csv';
    private array $headers = ['header_1', 'header_2', 'header_3', 'header_4'];
    private array $rows = [
        0 => ['row 1 content 1', 'row 1 content 2', 'row 1 content 3', 'row 1 content 4'],
        1 => ['row 2 content 1', 'row 2 content 2', 'row 2 content 3', 'row 2 content 4'],
    ];
    private CsvFile $csv;

    /**
     * @throws Exception
     */
    public function setUp(): void
    {
        // Delete test file
        if (file_exists($this->filename))
            unlink($this->filename);

        // Write test file
        $csv = new CsvFile();
        $csv->setHeaders($this->headers)->setRows($this->rows);
        SimpleCsv::write($this->filename, $csv);

        // Re-read for testing
        $this->csv = SimpleCsv::read($this->filename, true);
    }

    public function testGetHeaders()
    {
        $this->assertEquals($this->headers, $this->csv->getHeaders());
    }

    public function testGetRows()
    {
        $this->assertEquals($this->rows, $this->csv->getRows());
    }

    public function testGetRow()
    {
        $this->assertEquals($this->rows[0], $this->csv->getRow(0));
    }

    public function testGetRowsWithHeaders()
    {
        $rows = $this->csv->getRows(true);
        $keys = [];
        $values = [];
        foreach ($rows as $row) {
            $keys[] = array_keys($row);
            $values[] = array_values($row);
        }
        foreach ($keys as $row_keys) {
            $this->assertEquals($this->headers, $row_keys);
        }
        foreach ($values as $key => $row) {
            $this->assertEquals($this->rows[$key], $row);
        }
    }

    public function testGetRowWithHeaders()
    {
        $row = $this->csv->getRow(0, true);
        $keys = array_keys($row);
        $values = array_values($row);
        $this->assertEquals($keys, $this->headers);
        $this->assertEquals($values, $this->rows[0]);
    }

    /**
     * @throws Exception
     */
    public function testAddRowWithoutHeaders()
    {
        $row = ['row 3 content 1', 'row 3 content 2', 'row 3 content 3', 'row 3 content 4'];
        $this->csv->addRow($row);
        $this->assertEquals($row, $this->csv->getRow(2));
    }


    /**
     * @throws Exception
     */
    public function testAddRowWithHeaders()
    {
        $expected = [
            'row 3 content 1',
            'row 3 content 2',
            'row 3 content 3',
            'row 3 content 4'
        ];
        $row = [
            'header_2' => 'row 3 content 2',
            'header_4' => 'row 3 content 4',
            'header_1' => 'row 3 content 1',
            'header_3' => 'row 3 content 3'
        ];
        $this->csv->addRow($row);
        $this->assertEquals($expected, $this->csv->getRow(2));
    }

    /**
     * @throws Exception
     */
    public function testAddRowAddIndexWithoutHeaders()
    {
        $index = 1;
        $row = ['row 3 content 1', 'row 3 content 2', 'row 3 content 3', 'row 3 content 4'];
        $this->csv->addRowAtIndex($index, $row);
        $this->assertEquals($row, $this->csv->getRow($index));
    }

    /**
     * @throws Exception
     */
    public function testAddRowAtIndexWithHeaders()
    {
        $index = 1;
        $expected = [
            'row 3 content 1',
            'row 3 content 2',
            'row 3 content 3',
            'row 3 content 4'
        ];
        $row = [
            'header_2' => 'row 3 content 2',
            'header_4' => 'row 3 content 4',
            'header_1' => 'row 3 content 1',
            'header_3' => 'row 3 content 3'
        ];
        $this->csv->addRowAtIndex($index, $row);
        $this->assertEquals($expected, $this->csv->getRow($index));
    }

    /**
     * @throws Exception
     */
    public function testSetRowWithoutHeaders()
    {
        $index = 1;
        $row = ['row 3 content 1', 'row 3 content 2', 'row 3 content 3', 'row 3 content 4'];
        $this->csv->setRow($index, $row);
        $this->assertEquals($row, $this->csv->getRow($index));
    }

    /**
     * @throws Exception
     */
    public function testSetRowWithHeaders()
    {
        $index = 1;
        $expected = [
            'row 3 content 1',
            'row 3 content 2',
            'row 3 content 3',
            'row 3 content 4'
        ];
        $row = [
            'header_2' => 'row 3 content 2',
            'header_4' => 'row 3 content 4',
            'header_1' => 'row 3 content 1',
            'header_3' => 'row 3 content 3'
        ];
        $this->csv->setRow($index, $row);
        $this->assertEquals($expected, $this->csv->getRow($index));
    }

    /**
     * @throws Exception
     */
    public function testSetCell()
    {
        $rowIndex = 2;
        $columnIndex = 2;
        $newValue = "TEST";
        $this->csv->setCell($columnIndex, $rowIndex, $newValue);
        $this->assertEquals($newValue, $this->csv->getCell($columnIndex, $rowIndex));
    }

    /**
     * @throws Exception
     */
    public function testSetCellByHeaderName()
    {
        $rowIndex = 2;
        $columnName = 'header_2';
        $newValue = "TEST";
        $this->csv->setCellByHeader($columnName, $rowIndex, $newValue);
        $this->assertEquals($newValue, $this->csv->getCellByHeader($columnName, $rowIndex));
    }

    public function testDeleteRow()
    {
        $index = 0;
        $this->csv->deleteRow($index);
        $this->assertCount(1, $this->csv->getRows());
    }

    public function testSearch()
    {
        $value = $this->rows[1][2];
        $result = $this->csv->search($value);
        $this->assertEquals([$this->rows[1]], $result);
        $this->assertNotEquals([$this->rows[0]], $result);

        $result_found = $this->csv->search($value, true);
        $result_not_found = $this->csv->search(strtoupper($value), true);
        $this->assertEquals([$this->rows[1]], $result_found);
        $this->assertEquals([], $result_not_found);

        $result = $this->csv->search(substr($value, 0, 3), true, true);
        $this->assertEquals([], $result);

        $value = $this->rows[1][2];
        $result = $this->csv->search($value, false, false, true);
        $this->assertEquals($this->headers, $result[0]);
        $this->assertEquals($this->rows[1], $result[1]);
    }

    /**
     * @throws Exception
     */
    public function testSearchByColumnName()
    {
        $value = $this->rows[1][2];
        $column = "header_3";
        $result = $this->csv->searchByColumn($column, $value);
        $this->assertEquals([$this->rows[1]], $result);
        $this->assertNotEquals([$this->rows[0]], $result);
    }

    /**
     * @throws Exception
     */
    public function testSearchByColumnIndex()
    {
        $value = $this->rows[1][2];
        $column = 2;
        $result = $this->csv->searchByColumn($column, $value);
        $this->assertEquals([$this->rows[1]], $result);
        $this->assertNotEquals([$this->rows[0]], $result);
    }

}
