<?php

namespace Welford\Captcha;

use Exception;

class Provider {

    private string $chars;
    private int $length;
    private int $width;
    private int $height;
    private int $font_size;
    private string $captcha = "";
    private $image;

    /**
     * @throws Exception
     */
    public function __construct(string $validChars = null, int $length = 6, int $width = 200, int $height = 50, int $font_size = 28)
    {
        ($validChars !== null) ? $this->chars = $validChars : $this->chars = "123456789ABCDEFGHIJKLMNPQRSTUVQXYZabcdefghijklmnpqrstuvwxyz";
        $this->length = $length;
        $this->width = $width;
        $this->height = $height;
        $this->font_size = $font_size;
        $this->image = $this->make();
    }

    public function getCaptchaString(): string
    {
        return $this->captcha;
    }

    public function getCaptchaImage() {
        return $this->image;
    }

    /**
     * @throws Exception
     */
    private function make() {

        // Clear and create a new crypto secure captcha string
        $this->captcha = "";
        for($i = 0; $i < $this->length; $i++) {
            $random_character = $this->chars[random_int(0, strlen($this->chars) - 1)];
            $this->captcha .= $random_character;
        }

        // Get new image from background helper function
        $image = $this->generateBackground();

        // Apply black and white text using provided font files
        $black = imagecolorallocate($image, 0, 0, 0);
        $white = imagecolorallocate($image, 255, 255, 255);
        $textcolors = [$black, $white];

        // Load provided MIT fonts
        $fonts = [
            __DIR__ . '/fonts/Acme.ttf',
            __DIR__ . '/fonts/Ubuntu.ttf',
            __DIR__ . '/fonts/Merriweather.ttf',
            __DIR__ . '/fonts/PlayfairDisplay.ttf'
        ];

        // Add letters to image
        for($i = 0; $i < strlen($this->captcha); $i++) {
            $letter_space = ($this->width - 20)/strlen($this->captcha);
            $initial = 15;
            imagettftext($image, $this->font_size, rand(-10, 10), $initial + $i*$letter_space, rand($this->font_size, ($this->height - 10)), $textcolors[rand(0, 1)], $fonts[array_rand($fonts)], $this->captcha[$i]);
        }

        return $image;
    }

    private function generateBackground() {

        // Create new GdImage
        $image = imagecreatetruecolor($this->width, $this->height);
        imageantialias($image, true);

        $colors = [];

        $red = rand(125, 175);
        $green = rand(125, 175);
        $blue = rand(125, 175);

        // Create randomish colours
        for($i = 0; $i < 5; $i++) {
            $colors[] = imagecolorallocate($image, $red - 20*$i, $green - 20*$i, $blue - 20*$i);
        }

        // Fill image with randomish colours
        imagefill($image, 0, 0, $colors[0]);

        // Generate randomised background pattern
        for($i = 0; $i < 10; $i++) {
            imagesetthickness($image, rand(2, 10));
            $rect_color = $colors[rand(1, 4)];
            imagerectangle($image, rand(-10, ($this->width - 10)), rand(-10, 10), rand(-10, ($this->width - 10)), rand(($this->height - 10), ($this->height + 10)), $rect_color);
        }

        return $image;
    }

}
