<?php

namespace Welford\Captcha;

use Exception;
use GdImage;
use Google\ApiCore\ApiException;
use Google\ApiCore\ValidationException;
use Google\Cloud\TextToSpeech\V1\AudioConfig;
use Google\Cloud\TextToSpeech\V1\AudioEncoding;
use Google\Cloud\TextToSpeech\V1\SsmlVoiceGender;
use Google\Cloud\TextToSpeech\V1\SynthesisInput;
use Google\Cloud\TextToSpeech\V1\TextToSpeechClient;
use Google\Cloud\TextToSpeech\V1\VoiceSelectionParams;

class Provider {

    private string $chars;
    private int $length;
    private int $width;
    private int $height;
    private int $fontSize;
    private string $captcha = "";
    private GdImage $image;

    private TextToSpeechClient $textToSpeechClient;
    private string $audioOutput = "";
    private array $ttsSymbolMapping = [
        '#' => 'Hash',
        '&' => 'Ampersand',
        '!' => 'Exclamation Mark',
        '?' => 'Question Mark',
        '*' => 'Asterisk'
    ];

    /**
     * @throws ValidationException
     * @throws Exception
     */
    public function __construct(string $validChars = null, string $apiKeyFilePath = null, int $length = 6, int $width = 200, int $height = 50, int $fontSize = 28)
    {

        $chars = array_filter(str_split($validChars ?? ""));

        foreach($chars as $char) {
            if (!array_key_exists($char, $this->ttsSymbolMapping)) {
                throw new Exception("Your character list contained a symbol that is not supported. Valid symbols are " . implode('', array_keys($this->ttsSymbolMapping)) . '');
            }
        }

        ($validChars !== null) ? $this->chars = $validChars : $this->chars = "#&!?*123456789ABCDEFGHIJKLMNPQRSTUVQXYZabcdefghijklmnpqrstuvwxyz";
        $this->length = $length;
        $this->width = $width;
        $this->height = $height;
        $this->fontSize = $fontSize;
        $this->image = $this->make();

        if ($apiKeyFilePath !== null) {
            if (!file_exists($apiKeyFilePath)) {
                throw  new Exception("Authentication key file not found.");
            }
            $key = file_get_contents($apiKeyFilePath);
            $this->textToSpeechClient = new TextToSpeechClient(['credentials' => json_decode($key, true)]);
        }
    }

    public function getCaptchaString(): string
    {
        return $this->captcha;
    }

    public function getCaptchaImage() {
        return $this->image;
    }

    /**
     * @throws ApiException
     * @throws Exception
     */
    public function getCaptchaAudio(): string
    {
        if (isset($this->textToSpeechClient)) {
            if (!empty($this->audioOutput)) {
                return $this->audioOutput;
            }

            $split = str_split($this->captcha);
            $text = "<speak>";
            foreach ($split as $char) {
                if (ctype_digit($char)) {
                    $text .= "<p>" . $char . ".</p>";
                } else if (ctype_upper($char)) {
                    $text .= "<p>Upper case " . $char . ".</p>";
                } else if (ctype_lower($char)) {
                    $text .= "<p>Lower case " . $char . ".</p>";
                } else if (ctype_punct($char)) {
                    if (array_key_exists($char, $this->ttsSymbolMapping)) {
                        $text .= "<p>" . $this->ttsSymbolMapping[$char] . ".</p>";
                    }
                }
            }
            $text .= "</speak>";

            $input = new SynthesisInput();
            $input->setSsml($text);
            $voice = new VoiceSelectionParams();
            $voice->setLanguageCode('en-GB');
            $voice->setSsmlGender(SsmlVoiceGender::FEMALE);
            $audioConfig = new AudioConfig();
            $audioConfig->setAudioEncoding(AudioEncoding::MP3);
            $audio = $this->textToSpeechClient->synthesizeSpeech($input, $voice, $audioConfig);
            $this->audioOutput = base64_encode($audio->getAudioContent());
            return $this->audioOutput;
        }
        throw new Exception("The Google TTS service has not been initialised. Did you provide your API key file path?");
    }

    /**
     * @throws ApiException
     */
    public function getCaptchaAudioHtml(): string
    {
        $audioData = $this->getCaptchaAudio();
        return '<audio controls src="data:audio/mp3;base64,' . $audioData . '"></audio>';
    }

    /**
     * @throws Exception
     */
    private function make() {

        // Clear and create a new crypto secure captcha string
        $this->captcha = "";
        $this->audioOutput = "";
        for($i = 0; $i < $this->length; $i++) {
            $random_character = $this->chars[random_int(0, strlen($this->chars) - 1)];
            $this->captcha .= $random_character;
        }

        // Get new image from background helper function
        $image = $this->generateBackground();

        // Apply black and white text using provided font files
        $black = imagecolorallocate($image, 0, 0, 0);
        $white = imagecolorallocate($image, 255, 255, 255);
        $textcolors = [$black, $white];

        // Load provided MIT fonts
        $fonts = [
            __DIR__ . '/fonts/Acme.ttf',
            __DIR__ . '/fonts/Ubuntu.ttf',
            __DIR__ . '/fonts/Merriweather.ttf',
            __DIR__ . '/fonts/PlayfairDisplay.ttf'
        ];

        // Add letters to image
        for($i = 0; $i < strlen($this->captcha); $i++) {
            $letter_space = ($this->width - 20)/strlen($this->captcha);
            $initial = 15;
            imagettftext($image, $this->fontSize, rand(-10, 10), $initial + $i*$letter_space, rand($this->fontSize, ($this->height - 10)), $textcolors[rand(0, 1)], $fonts[array_rand($fonts)], $this->captcha[$i]);
        }

        return $image;
    }

    private function generateBackground() {

        // Create new GdImage
        $image = imagecreatetruecolor($this->width, $this->height);
        imageantialias($image, true);

        $colors = [];

        $red = rand(125, 175);
        $green = rand(125, 175);
        $blue = rand(125, 175);

        // Create randomish colours
        for($i = 0; $i < 5; $i++) {
            $colors[] = imagecolorallocate($image, $red - 20*$i, $green - 20*$i, $blue - 20*$i);
        }

        // Fill image with randomish colours
        imagefill($image, 0, 0, $colors[0]);

        // Generate randomised background pattern
        for($i = 0; $i < 10; $i++) {
            imagesetthickness($image, rand(2, 10));
            $rect_color = $colors[rand(1, 4)];
            imagerectangle($image, rand(-10, ($this->width - 10)), rand(-10, 10), rand(-10, ($this->width - 10)), rand(($this->height - 10), ($this->height + 10)), $rect_color);
        }

        return $image;
    }

}
