<?php

namespace Welford\WelfordTill\Util;

use Exception;
use Welford\WelfordTill\Models\Country;
use Welford\WelfordTill\Models\Option;
use Welford\WelfordTill\Models\OptionValue;
use Welford\WelfordTill\Models\Product;
use Welford\WelfordTill\Models\Zone;

class Utilities
{
    const ECOM_TYPE_OPENCART = 1;

    /**
     * @throws Exception
     */
    public static function productFromCommerceObject(array $object, float $tax = 0, float $tax_rate = 0, int $ecom_type = Utilities::ECOM_TYPE_OPENCART): Product
    {
        switch ($ecom_type) {
            case Utilities::ECOM_TYPE_OPENCART:
                $options = [];
                if (isset($object['options']) && is_array($object['options'])) {
                    foreach ($object['options'] as $option) {
                        $options[] = self::optionFromCommerceObject($option, $ecom_type);
                    }
                }
                return new Product(
                    $object['product_id'],
                    $object['sku'] != "" ? $object['sku'] : $object['model'],
                    html_entity_decode($object['name'] ?? '', ENT_QUOTES, 'UTF-8'),
                    (float)$object['special'] ? $object['special'] : $object['price'],
                    $tax,
                    $tax_rate,
                    $object['till_notes'] ?? '',
                    $options
                );
            default:
                throw new Exception('Unknown e-commerce type');
        }
    }

    /**
     * @throws Exception
     */
    public static function optionFromCommerceObject(array $object, int $ecom_type = Utilities::ECOM_TYPE_OPENCART): Option
    {
        switch ($ecom_type) {
            case Utilities::ECOM_TYPE_OPENCART:
                $option_values = [];
                if (isset($object['product_option_value']) && is_array($object['product_option_value'])) {
                    foreach ($object['product_option_value'] as $option_value) {
                        $option_values[] = self::optionValueFromCommerceObject($option_value, $ecom_type);
                    }
                }
                return new Option($object['product_option_id'], $object['option_id'], $object['name'], $object['type'], $object['value'], $object['required'], $option_values);
            default:
                throw new Exception('Unknown e-commerce type');
        }
    }

    /**
     * @throws Exception
     */
    public static function optionValueFromCommerceObject(array $object, int $ecom_type = Utilities::ECOM_TYPE_OPENCART): OptionValue
    {
        switch ($ecom_type) {
            case Utilities::ECOM_TYPE_OPENCART:
                return new OptionValue($object['product_option_value_id'], $object['option_value_id'], $object['name'], $object['price'], $object['price_prefix']);
            default:
                throw new Exception('Unknown e-commerce type');
        }
    }

    /**
     * @throws Exception
     */
    public static function countryFromCommerceObject(array $object, int $ecom_type = Utilities::ECOM_TYPE_OPENCART): Country
    {
        switch ($ecom_type) {
            case Utilities::ECOM_TYPE_OPENCART:
                return new Country($object['country_id'], $object['name'], $object['iso_code_2'], $object['iso_code_3']);
            default:
                throw new Exception('Unknown e-commerce type');
        }
    }

    /**
     * @throws Exception
     */
    public static function zoneFromCommerceObject(array $object, int $ecom_type = Utilities::ECOM_TYPE_OPENCART): Zone
    {
        switch ($ecom_type) {
            case Utilities::ECOM_TYPE_OPENCART:
                return new Zone($object['zone_id'], $object['name'], $object['country_id']);
            default:
                throw new Exception('Unknown e-commerce type');
        }
    }
}
